/** @internal *//** */
import { ev } from "../events";
import { ConfigObject } from "../../api/model";
import { Server } from "socket.io";
import osName from 'os-name';
import open from 'open';
import getPort from 'get-port';
import commandExists from 'command-exists';
import http from 'http'
import { processSendData } from "../../utils/tools";
import { log } from "../../logging/logging";

const 
placeholders = {
    "authenticated": "data:image/png;base64,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",
    "loading": "data:image/png;base64,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"
},
currentQrCodes = {
    "latest": placeholders.loading
},
serverSockets : any = {};

let io,
    express,
    app,
    gClient,
    PORT,
    server;

    const setUpApp = async () => {
    express = (await import('express')).default
    app = express();
    app.use(express.static(__dirname + '/node_modules'));  

    app.get('/', function(req, res) {  
        res.sendFile(__dirname + '/index.html');
    });
    
    
    app.get('/qr', function(req, res) {  
        const {sessionId} = req.query;
        const qr = sessionId ? currentQrCodes[sessionId] || currentQrCodes.latest : currentQrCodes.latest
        res.writeHead(200, { 'Content-Type': 'image/png' })
        res.write(Buffer.from(qr.replace('data:image/png;base64,', ''), 'base64'), 'utf8');
        res.end();
    });
    
}

export async function popup(config: ConfigObject) : Promise<string> {
    await setUpApp();
    const _p = process.env.PORT || config.port
    const preferredPort = typeof config.popup === "boolean" && config.popup && _p ? Number(_p) : config.popup;
    const popupListener = ev.on('**', async (data, sessionId, namespace) => {
        if(namespace?.includes("sessionData")) return;
        if (gClient) {
            await gClient.send({ data, sessionId, namespace });
            if(data?.includes && data?.includes("ready for account")) {
                await gClient.send({ data: config?.apiHost, sessionId, namespace: "ready" });
            }
        }
        if (namespace === 'qr') {
            currentQrCodes['latest'] = currentQrCodes[sessionId] = data;
        }
        if(data?.includes && data?.includes("Authenticated")) {
            currentQrCodes['latest'] = currentQrCodes[sessionId] = placeholders.authenticated
        }
        if(data?.includes && data?.includes("ready for account")) {
            //@ts-ignore
            popupListener.off()
            await closeHttp();
        }
    }, {objectify: true});

    /**
     * There should only be one instance of this open. If the server is already running, respond with the address.
     */

    if (server) return `http://localhost:${PORT}`;
    PORT = await getPort({ host: 'localhost', port: typeof preferredPort == 'number' ? [preferredPort, 7000, 7001, 7002] : [7000, 7001, 7002] });
    log.info(`popup port set to ${PORT}`)
    server = http.createServer(app);  
    if(!config?.qrPopUpOnly) {
        io = new Server(server)
        io.on('connection', function (client) {
            gClient = client;
            gClient.send({ data: 'CONNECTED', sessionId: config?.sessionId || 'session', namespace: 'SOCKET' })
            gClient.send({ data: config?.sessionId ? currentQrCodes[config?.sessionId] || currentQrCodes.latest : currentQrCodes.latest, sessionId: config?.sessionId || 'session', namespace: 'qr' })
        });
    }
    server.on("connection", (conn: any) => {
    const key = conn.remoteAddress + ':' + conn.remotePort;
    serverSockets[key] = conn;
        conn.on("close", () => {
            delete serverSockets[key];
        });
    });
    server.listen(PORT);
    processSendData({port:PORT})
    const os = osName();
    const appName = os.includes('macOS') ? 'google chrome' : os.includes('Windows') ? 'chrome' : 'google-chrome';
    const hasChrome = await commandExists(appName).then(()=>true).catch(()=>false);
    if(hasChrome){
        if(!config?.inDocker) await open(`http://localhost:${PORT}${config?.qrPopUpOnly?`/qr`:``}`, { app: {
        name: config?.executablePath || appName,    
        arguments: [ '--incognito']}, allowNonzeroExitCode: true}).catch(()=>{return;}); else return `http://localhost:${PORT}`;
    } else return `http://localhost:${PORT}${config?.qrPopUpOnly ? '/qr' : ''}`;

    return `http://localhost:${PORT}${config?.qrPopUpOnly ? '/qr' : ''}`
}

export const closeHttp = async () : Promise<void> => {
    if(!server) return;
  for (const key in serverSockets) {
    serverSockets[key].destroy();
  }
    return await new Promise(resolve => server.close(resolve))
}